/* 
 * log-tsd.c 
 *
 * Створює задану кількість (THREADS_NUM) потоків, кожен із яких 
 * відкриває власний журнальний файл і записує в нього повідомлення,
 * після чого завершує роботу. Покажчик на журнальний файл зберігається
 * в області приватних потокових даних. Закриття журнального файлу 
 * виконується через функцію очистки ключа.
 * Ілюструє порядок оперування приватними потоковими даними (функції
 * pthread_key_create(), pthread_setspecific(), pthread_getspecific()).
 *
 */

#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* Ключ доступу до покажчиків приватних журнальних файлів */
static pthread_key_t thread_log_key;

void write_to_thread_log(const char *message);
void close_thread_log(void *thread_log);
void *thread_function(void *arg);


/* Записує повідомлення message в приватний журнальний файл. */
void write_to_thread_log(const char *message)
{
        FILE *thread_log = (FILE *) pthread_getspecific(thread_log_key);

        if (thread_log != NULL)
                fputs(message, thread_log);
}

/* Функція-деструктор елемента області приватних потокових даних,
 * що містить покажчики приватних журнальних файлів. 
 * Закриває журнальний файл, на який указує thread_log. */
void close_thread_log(void *thread_log)
{
        if (thread_log != NULL)
                fclose((FILE*) thread_log);
}

/* Потокова функція */
void *thread_function(void *arg)
{
        enum { LOGNAME_LEN = 20 };
        char logname[LOGNAME_LEN];
        unsigned int thread_num = *((unsigned int *) arg);
        FILE *log;
        int terrno;

        /* Створює приватне ім'я журнального файлу. */
        snprintf(logname, LOGNAME_LEN, "thread%u.log", thread_num);
        /* Відкриває журнальний файл. */
        log = fopen(logname, "w");
        if (log == NULL) {
                fprintf(stderr, "Error opening thread log file %s\n",
                                                                logname);
                return NULL;
        }
        /* Зберігає покажчик журнального файлу в приватному блоці даних
           під ключем thread_log_key. */
        terrno = pthread_setspecific(thread_log_key, log);
        if (terrno != 0)
                fprintf(stderr, "Error saving private thread log pointer:"
                                                " %s\n", strerror(terrno));

        /* Записує початкове повідомлення в журнальний файл. */
        write_to_thread_log("Thread has started.\n");

        /* Далі повинна йти основна частина тіла потокової функції... */

        return NULL;
}

int main()
{
        enum { THREADS_NUM = 5 };
        pthread_t threads[THREADS_NUM];
        unsigned int thread_args[THREADS_NUM];
        int i, terrno;

        /* Створює ключ доступу до покажчиків приватних журнальних файлів.
           Ключ зберігається в глобальній змінній thread_log_key.
           close_thread_log() - функція очистки ключа */    
        terrno = pthread_key_create(&thread_log_key, close_thread_log);
        if (terrno != 0) {
                fprintf(stderr, "Error creating key: %s\n",
                                                strerror(terrno));
                exit(EXIT_FAILURE);
        }

        /* Створює потоки. */
        for (i = 0; i < THREADS_NUM; i++) {
                thread_args[i] = i;
                terrno = pthread_create(&(threads[i]), NULL,
                                thread_function, (void *) &thread_args[i]);
                if (terrno != 0) {
                        fprintf(stderr, "Error creating thread: %s\n",
                                                        strerror(terrno));
                        exit(EXIT_FAILURE);
                }
        }

        /* Чекає на завершення всіх потоків. */
        for (i = 0; i < THREADS_NUM; i++) {
                terrno = pthread_join(threads[i], NULL);
                if (terrno != 0) {
                        fprintf(stderr, "Error joining thread: %s\n",
                                                        strerror(terrno));
                        exit(EXIT_FAILURE);
                }
        }

        return EXIT_SUCCESS;
}
